<?php

namespace App\Services;

use App\Models\LedgerEntry;
use App\Models\Transaction;
use App\Models\Wallet;
use App\Models\Account;
use Illuminate\Support\Facades\DB;
use Exception;

class LedgerEngine
{
    /**
     * Perform a double-entry transaction.
     * All balance changes must go through this method.
     *
     * @param array $debits [account_id => amount]
     * @param array $credits [account_id => amount]
     * @param array $meta
     * @param string $transactionKey
     * @return Transaction
     * @throws Exception
     */
    public static function transact(array $debits, array $credits, array $meta, string $transactionKey): Transaction
    {
        return DB::transaction(function () use ($debits, $credits, $meta, $transactionKey) {
            // Idempotency check
            $existing = Transaction::where('transaction_key', $transactionKey)->first();
            if ($existing) {
                return $existing;
            }

            $totalDebit = array_sum($debits);
            $totalCredit = array_sum($credits);
            if (bccomp($totalDebit, $totalCredit, 2) !== 0) {
                throw new Exception('Ledger not balanced');
            }

            $transaction = Transaction::create([
                'company_id' => $meta['company_id'],
                'transaction_key' => $transactionKey,
                'user_id' => $meta['user_id'] ?? null,
                'type' => $meta['type'] ?? 'adjustment',
                'status' => 'processed',
                'amount' => $totalDebit,
                'fees' => $meta['fees'] ?? 0,
                'net_amount' => $totalDebit - ($meta['fees'] ?? 0),
                'description' => $meta['description'] ?? null,
                'created_by' => $meta['created_by'] ?? null,
                'processed_at' => now(),
                'processed_by' => $meta['processed_by'] ?? null,
                'reference_id' => $meta['reference_id'] ?? null,
            ]);

            // Ledger entries
            foreach ($debits as $accountId => $amount) {
                if ($amount <= 0) continue;
                LedgerEntry::create([
                    'company_id' => $meta['company_id'],
                    'transaction_id' => $transaction->id,
                    'account_id' => $accountId,
                    'user_id' => $meta['user_id'] ?? null,
                    'entry_type' => 'debit',
                    'amount' => $amount,
                    'entry_code' => uniqid('LDG'),
                    'description' => $meta['description'] ?? null,
                    'posted_at' => now(),
                ]);
                // Update wallet/account balance
                self::updateBalance($accountId, -$amount);
            }
            foreach ($credits as $accountId => $amount) {
                if ($amount <= 0) continue;
                LedgerEntry::create([
                    'company_id' => $meta['company_id'],
                    'transaction_id' => $transaction->id,
                    'account_id' => $accountId,
                    'user_id' => $meta['user_id'] ?? null,
                    'entry_type' => 'credit',
                    'amount' => $amount,
                    'entry_code' => uniqid('LDG'),
                    'description' => $meta['description'] ?? null,
                    'posted_at' => now(),
                ]);
                self::updateBalance($accountId, $amount);
            }
            return $transaction;
        });
    }

    /**
     * Update wallet/account balance in a transaction-safe way.
     * @param int $accountId
     * @param float $amount
     * @throws Exception
     */
    protected static function updateBalance(int $accountId, float $amount): void
    {
        $account = Account::findOrFail($accountId);
        if ($account->category === 'wallet') {
            $wallet = Wallet::where('account_id', $accountId)->lockForUpdate()->first();
            if (!$wallet) throw new Exception('Wallet not found');
            $newBalance = bcadd($wallet->balance, $amount, 2);
            if ($newBalance < 0) throw new Exception('Negative balance not allowed');
            $wallet->balance = $newBalance;
            $wallet->last_updated_at = now();
            $wallet->save();
        } else {
            // For non-wallet accounts, implement as needed
            // ...
        }
    }
}
