<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Models\TwoFactorAuthentication;
use Illuminate\Routing\Controller;

class TwoFactorController extends Controller
{
    public function show2faForm()
    {
        return view('auth.2fa');
    }

    public function verify2fa(Request $request)
    {
        $request->validate(['code' => 'required']);
        $user = Auth::user();
        $tfa = TwoFactorAuthentication::where('user_id', $user->id)->first();
        if ($tfa && $tfa->is_enabled) {
            $valid = false;
            // App-based (TOTP)
            if ($tfa->method === 'app' && $tfa->secret_key) {
                $valid = $this->verifyTOTP($request->code, $tfa->secret_key);
            }
            // SMS/email
            elseif (in_array($tfa->method, ['sms', 'email'])) {
                $valid = $request->code === $tfa->session_code;
            }
            if ($valid) {
                $request->session()->put('2fa_passed', true);
                return redirect()->intended('/dashboard');
            }
        }
        return back()->withErrors(['code' => 'Invalid 2FA code.']);
    }

    public function setupForm()
    {
        return view('auth.2fa_setup');
    }

    public function setup(Request $request)
    {
        $request->validate([
            'method' => 'required|in:sms,email,app',
            'phone_number' => 'nullable|string',
            'email' => 'nullable|email',
        ]);
        $user = Auth::user();
        $tfa = TwoFactorAuthentication::firstOrNew(['user_id' => $user->id]);
        $tfa->method = $request->method;
        $tfa->is_enabled = true;
        if ($request->method === 'sms') {
            $tfa->phone_number = $request->phone_number;
            $code = rand(100000, 999999);
            $tfa->session_code = $code;
            // TODO: Integrate SMS provider here
        } elseif ($request->method === 'email') {
            $tfa->email = $request->email;
            $code = rand(100000, 999999);
            $tfa->session_code = $code;
            // TODO: Integrate email provider here
        } elseif ($request->method === 'app') {
            $tfa->secret_key = $this->generateTOTPSecret();
        }
        $tfa->save();
        return redirect()->route('2fa.form')->with('status', '2FA setup complete.');
    }

    public function disable(Request $request)
    {
        $user = Auth::user();
        $tfa = TwoFactorAuthentication::where('user_id', $user->id)->first();
        if ($tfa) {
            $tfa->is_enabled = false;
            $tfa->save();
        }
        return redirect()->route('dashboard')->with('status', '2FA disabled.');
    }

    // TOTP (app-based) helpers
    private function generateTOTPSecret()
    {
        // Generate random base32 secret (for Google Authenticator, etc.)
        return bin2hex(random_bytes(10));
    }
    private function verifyTOTP($code, $secret)
    {
        // TODO: Integrate real TOTP verification (e.g. with spomky-labs/otphp or similar)
        return true; // Stub: always valid for now
    }
}
