<?php

namespace App\Http\Controllers;

use App\Models\Inventory;
use App\Models\Warehouse;
use App\Models\Product;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class InventoryController extends Controller
{
    public function index(Request $request)
    {
        $companyId = Auth::user()->company_id;
        $inventories = Inventory::with(['warehouse', 'product'])
            ->where('company_id', $companyId)
            ->paginate(20);
        return view('inventory.index', compact('inventories'));
    }

    public function create()
    {
        $warehouses = Warehouse::all();
        $products = Product::all();
        return view('inventory.create', compact('warehouses', 'products'));
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'warehouse_id' => 'required|exists:warehouses,id',
            'product_id' => 'required|exists:products,id',
            'quantity' => 'required|integer',
            'min_stock' => 'nullable|integer',
            'max_stock' => 'nullable|integer',
        ]);
        $data['company_id'] = Auth::user()->company_id;
        Inventory::create($data);
        return redirect()->route('inventory.index')->with('success', 'Inventory added.');
    }

    public function edit(Inventory $inventory)
    {
        $warehouses = Warehouse::all();
        $products = Product::all();
        return view('inventory.edit', compact('inventory', 'warehouses', 'products'));
    }

    public function update(Request $request, Inventory $inventory)
    {
        $data = $request->validate([
            'warehouse_id' => 'required|exists:warehouses,id',
            'product_id' => 'required|exists:products,id',
            'quantity' => 'required|integer',
            'min_stock' => 'nullable|integer',
            'max_stock' => 'nullable|integer',
        ]);
        $inventory->update($data);
        return redirect()->route('inventory.index')->with('success', 'Inventory updated.');
    }

    public function destroy(Inventory $inventory)
    {
        $inventory->delete();
        return redirect()->route('inventory.index')->with('success', 'Inventory deleted.');
    }
}
