<?php

namespace App\Http\Controllers\Install;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\URL;
use Illuminate\Support\Str;
use Illuminate\Routing\Controller;
use Exception;

class InstallerController extends Controller
{
    public function index()
    {
        if (File::exists(storage_path('installed.lock'))) {
            abort(403, 'Installer is locked.');
        }
        return view('install.index');
    }

    public function requirements(Request $request)
    {
        $requirements = [
            'php' => version_compare(PHP_VERSION, '8.2.0', '>='),
            'extensions' => [
                'openssl' => extension_loaded('openssl'),
                'pdo' => extension_loaded('pdo'),
                'mbstring' => extension_loaded('mbstring'),
                'tokenizer' => extension_loaded('tokenizer'),
                'xml' => extension_loaded('xml'),
                'ctype' => extension_loaded('ctype'),
                'json' => extension_loaded('json'),
                'fileinfo' => extension_loaded('fileinfo'),
            ],
        ];
        return response()->json($requirements);
    }

    public function permissions(Request $request)
    {
        $folders = [
            base_path('storage'),
            base_path('bootstrap/cache'),
            base_path('public/uploads'),
        ];
        $result = [];
        foreach ($folders as $folder) {
            $result[$folder] = File::isWritable($folder);
        }
        return response()->json($result);
    }

    public function database(Request $request)
    {
        $data = $request->validate([
            'host' => 'required',
            'port' => 'required',
            'database' => 'required',
            'username' => 'required',
            'password' => 'nullable',
        ]);
        $config = [
            'driver' => 'mysql',
            'host' => $data['host'],
            'port' => $data['port'],
            'database' => $data['database'],
            'username' => $data['username'],
            'password' => $data['password'],
        ];
        try {
            DB::purge('mysql');
            Config::set('database.connections.mysql', $config);
            DB::connection('mysql')->getPdo();
            return response()->json(['success' => true]);
        } catch (Exception $e) {
            return response()->json(['success' => false, 'error' => $e->getMessage()]);
        }
    }

    public function env(Request $request)
    {
        $env = $request->all();
        $envPath = base_path('.env');
        $envContent = File::get(base_path('.env.example'));
        foreach ($env as $key => $value) {
            $envContent = preg_replace("/^{$key}=.*$/m", "{$key}={$value}", $envContent);
        }
        File::put($envPath, $envContent);
        Artisan::call('config:clear');
        Artisan::call('cache:clear');
        return response()->json(['success' => true]);
    }

    public function migrate()
    {
        try {
            Artisan::call('migrate', ['--force' => true]);
            return response()->json(['success' => true]);
        } catch (Exception $e) {
            return response()->json(['success' => false, 'error' => $e->getMessage()]);
        }
    }

    public function seed()
    {
        try {
            Artisan::call('db:seed', ['--force' => true]);
            return response()->json(['success' => true]);
        } catch (Exception $e) {
            return response()->json(['success' => false, 'error' => $e->getMessage()]);
        }
    }

    public function key()
    {
        try {
            Artisan::call('key:generate', ['--force' => true]);
            return response()->json(['success' => true]);
        } catch (Exception $e) {
            return response()->json(['success' => false, 'error' => $e->getMessage()]);
        }
    }

    public function storage()
    {
        try {
            if (!File::exists(public_path('storage'))) {
                File::copyDirectory(storage_path('app/public'), public_path('storage'));
            }
            return response()->json(['success' => true]);
        } catch (Exception $e) {
            return response()->json(['success' => false, 'error' => $e->getMessage()]);
        }
    }

    public function lock()
    {
        File::put(storage_path('installed.lock'), 'installed:'.now());
        return response()->json(['success' => true]);
    }

    public function cron()
    {
        $commands = [
            '* * * * * php '.base_path('artisan').' schedule:run >> /dev/null 2>&1',
        ];
        return response()->json(['commands' => $commands]);
    }

    public function finish()
    {
        return redirect('/login');
    }
}
