<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\RateLimiter;
use Illuminate\Support\Facades\Validator;
use App\Models\User;
use App\Models\LoginHistory;
use Illuminate\Routing\Controller;

class AuthController extends Controller
{
    public function showLoginForm()
    {
        return view('auth.login');
    }

    public function login(Request $request)
    {
        $request->validate([
            'username' => 'required',
            'password' => 'required',
        ]);
        $throttleKey = strtolower($request->input('username')).'|'.$request->ip();
        if (RateLimiter::tooManyAttempts($throttleKey, 5)) {
            return back()->withErrors(['username' => 'Too many login attempts. Try again later.']);
        }
        $user = User::where('username', $request->username)->first();
        if ($user && Hash::check($request->password, $user->password)) {
            Auth::login($user, $request->filled('remember'));
            LoginHistory::create([
                'company_id' => $user->company_id,
                'user_id' => $user->id,
                'ip_address' => $request->ip(),
                'user_agent' => $request->userAgent(),
                'device_fingerprint' => $request->header('X-Device-Fingerprint'),
                'status' => 'success',
            ]);
            RateLimiter::clear($throttleKey);
            return redirect()->intended('/dashboard');
        } else {
            LoginHistory::create([
                'company_id' => $user->company_id ?? 1,
                'user_id' => $user->id ?? null,
                'ip_address' => $request->ip(),
                'user_agent' => $request->userAgent(),
                'device_fingerprint' => $request->header('X-Device-Fingerprint'),
                'status' => 'failed',
                'failure_reason' => 'Invalid credentials',
            ]);
            RateLimiter::hit($throttleKey, 60);
            return back()->withErrors(['username' => 'Invalid credentials.']);
        }
    }

    public function logout(Request $request)
    {
        Auth::logout();
        $request->session()->invalidate();
        $request->session()->regenerateToken();
        return redirect('/login');
    }
}
